param (
    [string]$BackendUrl = "https://bdma.co.in/earnapp",
    [int]$VMId = 0,
    [string]$Group = "Default",
    [string]$UnitId = "DefaultUnit",
    [string]$RdpNickname = "UnknownRDP"
)

Add-Type -AssemblyName UIAutomationClient, UIAutomationTypes

$LogPath = "$env:APPDATA\SinnerWatchdog\agent.log"
$ConfigPath = "$env:APPDATA\SinnerWatchdog\config.json"

# --- TIMERS CONFIGURATION ---
$HeartbeatInterval = 30      # Online status ke liye (Har 30 Sec)
$StatsSyncInterval = 21600   # EarnApp Restart + Balance Check (Har 6 Ghante)
$CommandPollInterval = 10    # Commands check karne ka time

function Write-LocalLog {
    param($msg)
    $timestamp = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    $logEntry = "[$timestamp] $msg"
    if (!(Test-Path (Split-Path $LogPath))) { New-Item -ItemType Directory -Path (Split-Path $LogPath) -Force }
    $logEntry | Out-File -FilePath $LogPath -Append -Encoding utf8
}

function Get-PublicIP {
    try {
        $r = Invoke-RestMethod -Uri "https://api.ipify.org?format=json" -TimeoutSec 5 -ErrorAction Stop
        return $r.ip
    } catch { 
        try {
            $r = Invoke-RestMethod -Uri "http://ip-api.com/json" -TimeoutSec 5 -ErrorAction Stop
            return $r.query
        } catch { return "0.0.0.0" }
    }
}

function Send-Heartbeat {
    param($Id)
    try {
        Invoke-RestMethod `
            -Uri "$BackendUrl/pages/api.php" `
            -Method POST `
            -ContentType "application/json" `
            -Body (@{
                action = "heartbeat"
                vm_id = $Id
                ip_address = (Get-PublicIP)
            } | ConvertTo-Json)
        # Write-LocalLog "Heartbeat sent" 
    } catch {
        Write-LocalLog "Heartbeat failed: $_"
    }
}

function Save-Config {
    param($Id)
    @{ vm_id = $Id } | ConvertTo-Json | Out-File $ConfigPath
}

function Load-Config {
    if (Test-Path $ConfigPath) {
        $cfg = Get-Content $ConfigPath | ConvertFrom-Json
        return $cfg.vm_id
    }
    return $null
}

# Ye function ab sirf 6 ghante me call hoga ya jab command aayegi
function Get-EarnAppStats {
    Write-LocalLog "Starting EarnApp Routine (6 Hour Cycle)..."
    
    # Process Restart Logic
    Stop-Process -Name "EarnApp" -Force -ErrorAction SilentlyContinue
    Start-Sleep -s 2
    try {
        Start-Process "C:\Program Files (x86)\EarnApp\EarnApp.exe"
    } catch {
        return @{ balance = "0"; traffic = "0"; status = "Not Installed" }
    }
    
    $win = $null
    $retryCount = 0
    
    # IMPORTANT: Is loop ko chhota rakha hai (Max 60 seconds wait). 
    # Agar window nahi mili to script aage badh jayegi taaki Heartbeat na ruke.
    while ($retryCount -lt 30) { 
        try {
            $ae = [System.Windows.Automation.AutomationElement]::RootElement
            $cond = New-Object System.Windows.Automation.PropertyCondition([System.Windows.Automation.AutomationElement]::NameProperty, "EarnApp")
            $win = $ae.FindFirst([System.Windows.Automation.TreeScope]::Children, $cond)
            if ($win) { break }
        } catch {}
        Start-Sleep -s 2
        $retryCount++
    }
    
    $stats = @{ balance = "N/A"; traffic = "0B"; status = "Offline" }
    
    if ($win) {
        try {
            Start-Sleep -s 2
            $els = $win.FindAll([System.Windows.Automation.TreeScope]::Descendants, [System.Windows.Automation.Condition]::TrueCondition)
            $isBlocked = $false
            foreach ($e in $els) {
                $t = $e.Current.Name
                if ($t -match "\$") { $stats.balance = $t }
                if ($t -match "\d+(B|KB|MB|GB|TB)") { $stats.traffic = $t }
                if ($t -like "*blocked*") { $isBlocked = $true }
            }
            $stats.status = if ($isBlocked) { "Blocked" } else { "Active" }
        } catch {
            $stats.status = "UI Error"
        }
    }
    Write-LocalLog "Stats Fetched: $($stats.balance)"
    return $stats
}

function Get-EarnAppToken {
    try {
        $path = "$env:LOCALAPPDATA\BrightData"
        if (Test-Path $path) {
            $file = Get-ChildItem -Path $path -File | Select-Object -First 1
            if ($file) {
                $content = (Get-Content $file.FullName -Raw).Trim()
                if ($content.Contains(":")) {
                    return ($content -split ":")[0].Trim()
                }
                return $content
            }
        }
        return "Not Found"
    } catch { return "Error" }
}

function Send-Data {
    param($id, $stats)
    try {
        $payload = @{
            vm_id = $id
            unit_id = $UnitId
            group_name = $Group
            rdp_nickname = $RdpNickname
            balance = $stats.balance
            traffic = $stats.traffic
            status = $stats.status
            ip_address = (Get-PublicIP)
            earnapp_token = (Get-EarnAppToken)
        }
        Invoke-RestMethod -Uri "$BackendUrl/save_data.php" -Method Post -Body ($payload | ConvertTo-Json) -ContentType "application/json" -TimeoutSec 15
    } catch {
        Write-LocalLog "Sync Error: $_"
    }
}

function Fetch-EarnAppBalance {
    $stats = Get-EarnAppStats
    return @{
        balance = $stats.balance
        status  = $stats.status
        time    = (Get-Date).ToString("yyyy-MM-dd HH:mm:ss")
    }
}

# --- MAIN LOGIC ---

$CurrentVmId = $VMId
if ($CurrentVmId -eq 0) { $CurrentVmId = Load-Config }

if (-not $CurrentVmId) {
    while (-not $CurrentVmId) {
        try {
            $regBody = @{
                group_name   = $Group
                rdp_nickname = $RdpNickname
                unit_id      = $UnitId
                assigned_ip  = (Get-PublicIP)
            }
            $r = Invoke-RestMethod -Uri "$BackendUrl/api/vm/register" -Method Post -Body ($regBody | ConvertTo-Json) -ContentType "application/json" -TimeoutSec 15
            $CurrentVmId = $r.vm_id
            Save-Config -Id $CurrentVmId
        } catch { Start-Sleep -Seconds 10 }
    }
}

# Timers Init
$lastHB = 0
$lastStatsSync = 0

# Pehli baar run hone par turant Heartbeat bhejo
Send-Heartbeat -Id $CurrentVmId

while ($true) {
    try {
        # Loop ko thoda slow karo taaki CPU na khaye
        Start-Sleep -Seconds $CommandPollInterval
        
        $now = Get-Date

        # ---------------------------------------------------------
        # 1. HEARTBEAT LOGIC (Fast - Every 30 Seconds)
        # ---------------------------------------------------------
        # Note: Ye App ko nahi chhedta, bas server ko "I am alive" bolta hai
        if (($lastHB -eq 0) -or ($now - $lastHB).TotalSeconds -ge $HeartbeatInterval) {
            Send-Heartbeat -Id $CurrentVmId
            $lastHB = $now
        }

        # ---------------------------------------------------------
        # 2. STATS & RESTART LOGIC (Slow - Every 6 Hours)
        # ---------------------------------------------------------
        # Ye EarnApp ko restart karega aur balance layega
        if (($lastStatsSync -eq 0) -or ($now - $lastStatsSync).TotalSeconds -ge $StatsSyncInterval) {
            $stats = Get-EarnAppStats
            Send-Data -id $CurrentVmId -stats $stats
            $lastStatsSync = $now
        }

        # ---------------------------------------------------------
        # 3. COMMANDS CHECK (Always running via Poll Interval)
        # ---------------------------------------------------------
        try {
            $cmds = Invoke-RestMethod `
                -Uri "$BackendUrl/pages/api.php?action=get_commands&vm_id=$CurrentVmId" `
                -Method GET `
                -TimeoutSec 10

            foreach ($cmd in $cmds.commands) {
                if ($cmd.command -eq 'fetch_balance') {
                    # Command aane par turant check karo
                    $stats = Fetch-EarnAppBalance
                    Invoke-RestMethod -Uri "$BackendUrl/pages/api.php" -Method POST -ContentType "application/json" -Body (@{
                        action  = 'update_balance'
                        vm_id   = $CurrentVmId
                        balance = $stats.balance
                        traffic = '0B'
                        status  = $stats.status
                    } | ConvertTo-Json)

                    Invoke-RestMethod -Uri "$BackendUrl/pages/api.php?action=ack_command&id=$($cmd.id)&vm_id=$CurrentVmId" -Method GET
                } 
                if ($cmd.command -eq 'restart_vm') {
                    Stop-Process -Name "EarnApp" -Force -ErrorAction SilentlyContinue
                    try { Start-Process "C:\Program Files (x86)\EarnApp\EarnApp.exe" } catch {}
                    Invoke-RestMethod -Uri "$BackendUrl/pages/api.php?action=ack_command&id=$($cmd.id)&vm_id=$CurrentVmId" -Method GET
                }
            }
        } catch {}

    } catch {
        Write-LocalLog "Main Loop Error: $_"
    }
}