param (
    [string]$BackendUrl = "https://bdma.co.in/earnapp",
    [int]$VMId = 0,
    [string]$Group = "Default",
    [string]$UnitId = "DefaultUnit",
    [string]$RdpNickname = "UnknownRDP"
)

# Strip trailing slash from BackendUrl to avoid double-slash URLs
$BackendUrl = $BackendUrl.TrimEnd('/')

Set-StrictMode -Off
[System.Net.ServicePointManager]::DnsRefreshTimeout = 0
[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
Add-Type -AssemblyName UIAutomationClient, UIAutomationTypes

$LogPath    = "$env:APPDATA\SinnerWatchdog\agent.log"
$ConfigPath = "$env:APPDATA\SinnerWatchdog\config.json"

$HeartbeatInterval   = 60
$StatsSyncInterval   = 300
$CommandPollInterval = 10
$LogCleanupInterval  = 300
$lastLogCleanup      = Get-Date

# --- Logging ---
function Write-LocalLog {
    param($msg)
    $ts = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    if (!(Test-Path (Split-Path $LogPath))) {
        New-Item -ItemType Directory -Path (Split-Path $LogPath) -Force | Out-Null
    }
    "[$ts] $msg" | Out-File $LogPath -Append -Encoding utf8
}

# --- HTTP helper  no external dependencies ---
function Invoke-RestMethodWithRetry {
    param(
        [string]$Uri,
        [string]$Method      = "GET",
        [string]$ContentType = "application/json",
        [string]$Body        = "",
        [int]$TimeoutSec     = 15,
        [int]$MaxRetries     = 3,
        [int]$InitialDelaySec = 5
    )
    $attempt = 0
    while ($attempt -lt $MaxRetries) {
        try {
            if ($Method -eq "GET") {
                return Invoke-RestMethod -Uri $Uri -Method GET -TimeoutSec $TimeoutSec -DisableKeepAlive
            } else {
                return Invoke-RestMethod -Uri $Uri -Method $Method `
                    -ContentType $ContentType -Body $Body `
                    -TimeoutSec $TimeoutSec -DisableKeepAlive
            }
        } catch {
            $attempt++
            Write-LocalLog "HTTP attempt $attempt/$MaxRetries failed for $Uri : $($_.Exception.Message)"
            if ($attempt -lt $MaxRetries) {
                $delay = $InitialDelaySec * [Math]::Pow(2, $attempt - 1)
                Start-Sleep -Seconds ([int]$delay)
            } else {
                throw $_.Exception
            }
        }
    }
}

# --- Public IP ---
function Get-PublicIP {
    try { (Invoke-RestMethod "https://api.ipify.org?format=json" -TimeoutSec 5).ip }
    catch {
        try { (Invoke-RestMethod "http://ip-api.com/json" -TimeoutSec 5).query }
        catch { "0.0.0.0" }
    }
}

# --- Config persistence ---
function Save-Config {
    param($Id)
    @{ vm_id = $Id; unit_id = $UnitId; group = $Group; rdp = $RdpNickname } |
        ConvertTo-Json | Out-File $ConfigPath -Force
}

function Load-Config {
    if (Test-Path $ConfigPath) {
        $cfg = (Get-Content $ConfigPath | ConvertFrom-Json)
        return $cfg.vm_id
    }
    return $null
}

# --- Self-stop ---
function Self-Stop {
    Write-LocalLog "stop_agent command received  shutting down agent"
    $taskName = "SinnerWatchdogAgent"
    Stop-ScheduledTask  -TaskName $taskName -ErrorAction SilentlyContinue
    Unregister-ScheduledTask -TaskName $taskName -Confirm:$false -ErrorAction SilentlyContinue
    if (Test-Path $ConfigPath) { Remove-Item $ConfigPath -Force -ErrorAction SilentlyContinue }
    Write-LocalLog "Agent stopped and unregistered"
    exit 0
}

# --- EarnApp UI automation ---
function Get-EarnAppStatus {
    param($elements)
    foreach ($e in $elements) {
        $t = $e.Current.Name
        if ($t -like "*Earning income*")           { return "Running" }
        if ($t -like "*Blocked*")                  { return "Blocked" }
        if ($t -like "*resource sharing paused*")  { return "Paused" }
        if ($t -like "*Connecting*")               { return "Connecting" }
        if ($t -like "*Resume earning*")           { return "Paused" }
    }
    return $null
}

function Minimize-Window {
    param($hwnd)
    try {
        Add-Type -TypeDefinition @"
using System;
using System.Runtime.InteropServices;
public class WinHelper {
    [DllImport("user32.dll")]
    public static extern bool ShowWindow(IntPtr hWnd, int nCmdShow);
    [DllImport("user32.dll")]
    public static extern IntPtr FindWindow(string lpClassName, string lpWindowName);
}
"@ -ErrorAction SilentlyContinue
        [WinHelper]::ShowWindow($hwnd, 2) | Out-Null # 2 = SW_MINIMIZE
    } catch {}
}

function Find-EarnAppWindow {
    try {
        $ae   = [System.Windows.Automation.AutomationElement]::RootElement
        $cond = New-Object System.Windows.Automation.PropertyCondition(
            [System.Windows.Automation.AutomationElement]::NameProperty, "EarnApp"
        )
        $win = $ae.FindFirst([System.Windows.Automation.TreeScope]::Children, $cond)
        if ($win) {
            try {
                $hwnd = $win.Current.NativeWindowHandle
                if ($hwnd) { Minimize-Window $hwnd }
            } catch {}
        }
        return $win
    } catch { return $null }
}

function Get-EarnAppBalanceFromFile {
    # Read balance directly from EarnApp local data files - no window needed
    try {
        $paths = @(
            "$env:LOCALAPPDATA\EarnApp",
            "$env:APPDATA\EarnApp",
            "$env:LOCALAPPDATA\Programs\EarnApp"
        )
        foreach ($p in $paths) {
            if (Test-Path $p) {
                $files = Get-ChildItem $p -Filter "*.json" -ErrorAction SilentlyContinue
                foreach ($f in $files) {
                    try {
                        $j = Get-Content $f.FullName -Raw | ConvertFrom-Json
                        if ($j.balance -ne $null) { return [decimal]$j.balance }
                        if ($j.current_balance -ne $null) { return [decimal]$j.current_balance }
                    } catch {}
                }
            }
        }
    } catch {}
    return $null
}

function Get-EarnAppStatusFromProcess {
    # Check if EarnApp process is running at all
    $proc = Get-Process -Name "EarnApp" -ErrorAction SilentlyContinue
    if ($proc) { return "Running" }
    return $null
}

function Get-EarnAppStats {
    Write-LocalLog "EarnApp stats started"
    $stats = @{ balance = 0; traffic = "0B"; status = "Offline" }

    $win = Find-EarnAppWindow

    if (-not $win) {
        Write-LocalLog "EarnApp window not found - trying background read"

        # Try to get status/balance without window
        $procStatus = Get-EarnAppStatusFromProcess
        $fileBalance = Get-EarnAppBalanceFromFile

        if ($procStatus -or $fileBalance -ne $null) {
            if ($fileBalance -ne $null) { $stats.balance = $fileBalance }
            $stats.status = if ($procStatus) { $procStatus } else { "Offline" }
            Write-LocalLog "Background read OK | Balance=$($stats.balance) | Status=$($stats.status)"
            return $stats
        }

        # Background read failed - try restart
        Write-LocalLog "Background read failed - waiting 30s then restart"
        Start-Sleep 30
        $win = Find-EarnAppWindow

        if (-not $win) {
            Write-LocalLog "Still not found - restarting EarnApp"
            Stop-Process -Name EarnApp -Force -ErrorAction SilentlyContinue
            Start-Sleep 2
            try { Start-Process "C:\Program Files (x86)\EarnApp\EarnApp.exe" -WindowStyle Minimized } catch {}
            Start-Sleep 40
            $win = Find-EarnAppWindow

            if (-not $win) {
                # Final fallback - return last known process status
                $stats.status = if (Get-Process -Name "EarnApp" -ErrorAction SilentlyContinue) { "Running" } else { "Offline" }
                Write-LocalLog "EarnApp unreachable - reporting status: $($stats.status)"
                return $stats
            }
        }
    }

    # --- Read stats from window ---
    function Read-StatsFromWindow {
        param($window)
        $s = @{ balance = 0; status = "Offline" }
        try {
            $els = $window.FindAll(
                [System.Windows.Automation.TreeScope]::Descendants,
                [System.Windows.Automation.Condition]::TrueCondition
            )
            foreach ($e in $els) {
                $t = $e.Current.Name
                if ($t -match "\$") {
                    $parsed = [decimal]($t -replace '[^0-9\.]', '')
                    if ($parsed -gt 0) { $s.balance = $parsed }
                }
            }
            $det = Get-EarnAppStatus -elements $els
            $s.status = if ($det) { $det } else { "Offline" }
        } catch {
            $s.status = "Offline"
        }
        return $s
    }

    try {
        if ($win) {
            $read = Read-StatsFromWindow $win
            $stats.balance = $read.balance
            $stats.status  = $read.status

            # --- Connecting mode handler ---
            if ($stats.status -eq "Connecting") {
                Write-LocalLog "Status=Connecting - waiting for app to stabilize (max 5 min)"
                $connectStart  = Get-Date
                $connectMaxSec = 300   # 5 minutes total timeout
                $pollSec       = 30    # check every 30s

                while ($stats.status -eq "Connecting") {
                    $elapsed = (Get-Date) - $connectStart

                    # Timeout reached - restart EarnApp
                    if ($elapsed.TotalSeconds -ge $connectMaxSec) {
                        Write-LocalLog "Connecting timeout ($connectMaxSec s) - restarting EarnApp"
                        Stop-Process -Name EarnApp -Force -ErrorAction SilentlyContinue
                        Start-Sleep 2
                        try { Start-Process "C:\Program Files (x86)\EarnApp\EarnApp.exe" -WindowStyle Minimized } catch {}
                        Start-Sleep 40
                        $win = Find-EarnAppWindow
                        if ($win) {
                            $read = Read-StatsFromWindow $win
                            $stats.balance = $read.balance
                            $stats.status  = $read.status
                            # If still Connecting after restart - report Offline
                            if ($stats.status -eq "Connecting") {
                                Write-LocalLog "Still Connecting after restart - reporting Offline"
                                $stats.status = "Offline"
                            }
                        } else {
                            $stats.status = "Offline"
                        }
                        break
                    }

                    Write-LocalLog "Still Connecting ($([int]$elapsed.TotalSeconds)s elapsed) - waiting ${pollSec}s more"
                    Start-Sleep $pollSec

                    # Re-read window
                    $win = Find-EarnAppWindow
                    if ($win) {
                        $read = Read-StatsFromWindow $win
                        $stats.balance = $read.balance
                        $stats.status  = $read.status
                        Write-LocalLog "Recheck: Status=$($stats.status) Balance=$($stats.balance)"
                    } else {
                        Write-LocalLog "Window lost during Connecting wait - reporting Offline"
                        $stats.status = "Offline"
                        break
                    }
                }
            }
            # --- End Connecting handler ---

        } else {
            $stats.status = "Offline"
        }
    } catch {
        $stats.status = "Offline"
        Write-LocalLog "EarnApp stats error: $($_.Exception.Message)"
    }

    Write-LocalLog "EarnApp stats done | Balance=$($stats.balance) | Status=$($stats.status)"
    return $stats
}

function Get-EarnAppToken {
    try {
        $path = "$env:LOCALAPPDATA\BrightData"
        if (Test-Path $path) {
            $file = Get-ChildItem $path -File | Select-Object -First 1
            if ($file) {
                $c = (Get-Content $file.FullName -Raw).Trim()
                if ($c.Contains(":")) { ($c -split ":")[0].Trim() } else { $c }
            } else { "Not Found" }
        } else { "Not Found" }
    } catch { "Error" }
}

function Ensure-EarnAppReady {
    Write-LocalLog "Ensure-EarnAppReady started"
    Stop-Process -Name EarnApp -Force -ErrorAction SilentlyContinue
    Start-Sleep 2
    try { Start-Process "C:\Program Files (x86)\EarnApp\EarnApp.exe" -WindowStyle Minimized } catch {
        Write-LocalLog "Failed to start EarnApp"
    }
    Start-Sleep 15
    $win = Find-EarnAppWindow
    if ($win) { Write-LocalLog "EarnApp window ready"; return $true }
    else { Write-LocalLog "EarnApp window not found after restart"; return $false }
}

# --- Data send ---
function Send-Data {
    param($id, $stats)
    try {
        Invoke-RestMethodWithRetry "$BackendUrl/save_data.php" -Method POST `
            -ContentType "application/json" -TimeoutSec 15 `
            -Body (@{
                vm_id         = $id
                unit_id       = $UnitId
                group_name    = $Group
                rdp_nickname  = $RdpNickname
                balance       = ("{0:F4}" -f [decimal]$stats.balance)
                traffic       = $stats.traffic
                status        = $stats.status
                ip_address    = (Get-PublicIP)
                earnapp_token = (Get-EarnAppToken)
            } | ConvertTo-Json)
    } catch {
        Write-LocalLog "Sync Error: $($_.Exception.Message)"
    }
}

# --- Main  Registration ---
$CurrentVmId = if ($VMId -ne 0) { $VMId } else { Load-Config }

if (-not $CurrentVmId) {
    Write-LocalLog "No VM ID found  registering with server"
    while (-not $CurrentVmId) {
        try {
            $r = Invoke-RestMethodWithRetry "$BackendUrl/api/vm/register.php" `
                -Method POST -ContentType "application/json" -TimeoutSec 15 `
                -Body (@{
                    group_name    = $Group
                    rdp_nickname  = $RdpNickname
                    unit_id       = $UnitId
                    assigned_ip   = (Get-PublicIP)
                    earnapp_token = (Get-EarnAppToken)
                } | ConvertTo-Json)

            if ($r.success) {
                $CurrentVmId = $r.vm_id
                Save-Config $CurrentVmId
                Write-LocalLog "Registered as VM ID $CurrentVmId (bridged=$($r.bridged -eq $true))"
            } else {
                Write-LocalLog "Registration failed: $($r.error)"
                Start-Sleep 15
            }
        } catch {
            Write-LocalLog "VM registration failed: $($_.Exception.Message)"
            Start-Sleep 10
        }
    }
}

# --- Main  Event loop ---
$lastHeartbeat   = [DateTime]::MinValue
$lastStatsSync   = [DateTime]::MinValue
$lastCommandPoll = [DateTime]::MinValue

Write-LocalLog "Agent started  VM ID: $CurrentVmId"
Write-LocalLog "Initial stats sync on startup"
$stats = Get-EarnAppStats
Send-Data $CurrentVmId $stats

while ($true) {
    Start-Sleep 5
    $now = Get-Date

    # --- Command poll ---
    if (($now - $lastCommandPoll).TotalSeconds -ge $CommandPollInterval) {
        try {
            $cmd = Invoke-RestMethodWithRetry "$BackendUrl/pages/api.php?action=fetch_command&vm_id=$CurrentVmId" -TimeoutSec 10

            if ($cmd -and $cmd.action) {
                Write-LocalLog "Command received: $($cmd.action)"

                if ($cmd.action -eq "stop_agent") {
                    Invoke-RestMethodWithRetry "$BackendUrl/pages/api.php?action=ack_command&id=$($cmd.id)&vm_id=$CurrentVmId" -TimeoutSec 10
                    Self-Stop
                }
                elseif ($cmd.action -eq "restart_vm") {
                    Write-LocalLog "VM restart command received"
                    Invoke-RestMethodWithRetry "$BackendUrl/pages/api.php?action=ack_command&id=$($cmd.id)&vm_id=$CurrentVmId" -TimeoutSec 10
                    Start-Process "shutdown.exe" -ArgumentList "/r /t 5 /f" -WindowStyle Hidden
                }
                elseif ($cmd.action -eq "fetch_balance") {
                    Write-LocalLog "Fetch balance command received"
                    if (Ensure-EarnAppReady) {
                        Start-Sleep 20
                        $stats = Get-EarnAppStats
                        Send-Data $CurrentVmId $stats
                    }
                    Invoke-RestMethodWithRetry "$BackendUrl/pages/api.php?action=command_result" `
                        -Method POST -ContentType "application/json" `
                        -Body (@{ command_id = $cmd.id; success = $true; message = "Balance fetched" } | ConvertTo-Json)
                    Invoke-RestMethodWithRetry "$BackendUrl/pages/api.php?action=ack_command&id=$($cmd.id)&vm_id=$CurrentVmId" -TimeoutSec 10
                }
                elseif ($cmd.action -eq "fetch_balance_global") {
                    Write-LocalLog "Group balance command received"
                    if (Ensure-EarnAppReady) {
                        $stats = Get-EarnAppStats
                        Send-Data $CurrentVmId $stats
                    }
                    Invoke-RestMethodWithRetry "$BackendUrl/pages/api.php?action=ack_command&id=$($cmd.id)&vm_id=$CurrentVmId" -TimeoutSec 10
                }
            }
        } catch {
            Write-LocalLog "Command poll failed: $($_.Exception.Message)"
        }
        $lastCommandPoll = $now
    }

    # --- Heartbeat ---
    if (($now - $lastHeartbeat).TotalSeconds -ge $HeartbeatInterval) {
        try {
            Invoke-RestMethodWithRetry "$BackendUrl/save_data.php" -Method POST `
                -ContentType "application/json" `
                -Body (@{ action = "heartbeat"; vm_id = $CurrentVmId; ip_address = (Get-PublicIP) } | ConvertTo-Json)
            Write-LocalLog "Heartbeat sent - VM $CurrentVmId"
        } catch {
            Write-LocalLog "Heartbeat failed: $($_.Exception.Message)"
        }
        $lastHeartbeat = $now
    }

    # --- Stats sync ---
    if (($now - $lastStatsSync).TotalSeconds -ge $StatsSyncInterval) {
        $stats = Get-EarnAppStats
        Send-Data $CurrentVmId $stats
        $lastStatsSync = $now
    }

    # --- Log cleanup ---
    if (($now - $lastLogCleanup).TotalSeconds -ge $LogCleanupInterval) {
        try {
            if (Test-Path $LogPath)   { Clear-Content $LogPath -ErrorAction SilentlyContinue }
            if (Test-Path $ConfigPath) { Remove-Item $ConfigPath -Force -ErrorAction SilentlyContinue }
        } catch {}
        $lastLogCleanup = $now
    }
}