<#
.SYNOPSIS
    SinnerWatchdog Installer
    - Downloads Agent
    - Sets up Persistence (Scheduled Task)
#>

param (
    [string]$BackendUrl,
    [int]$VMId = 0,
    [string]$Group,
    [string]$UnitId,
    [string]$RdpNickname = "UnknownRDP"
)

Write-Host "`n========================================" -ForegroundColor Cyan
Write-Host "  SinnerWatchdog Agent Installer" -ForegroundColor Cyan
Write-Host "========================================`n" -ForegroundColor Cyan

$AgentDir = "$env:APPDATA\SinnerWatchdog"
$AgentPath = "$AgentDir\agent.ps1"
$TaskName = "SinnerWatchdogAgent"
$ErrorOccurred = $false

# 1. Create Directory
Write-Host "[1/4] Creating agent directory..." -ForegroundColor Yellow
try {
    if (-not (Test-Path $AgentDir)) { 
        New-Item -ItemType Directory -Path $AgentDir | Out-Null 
        Write-Host "      Created: $AgentDir" -ForegroundColor Green
    } else {
        Write-Host "      Directory already exists" -ForegroundColor Green
    }
} catch {
    Write-Host "      Failed to create directory: $_" -ForegroundColor Red
    $ErrorOccurred = $true
}

# 2. Download Agent
Write-Host "`n[2/4] Downloading agent script..." -ForegroundColor Yellow
try {
    Invoke-WebRequest -Uri "$BackendUrl/static/agent.ps1" -OutFile $AgentPath -ErrorAction Stop
    Write-Host "      Agent downloaded successfully" -ForegroundColor Green
} catch {
    Write-Host "      Failed to download agent: $_" -ForegroundColor Red
    $ErrorOccurred = $true
}

# 3. Create Scheduled Task (Persistence)
if (-not $ErrorOccurred) {
    Write-Host "`n[3/4] Setting up scheduled task..." -ForegroundColor Yellow
    try {
        # UPDATED: Now passes Group and RdpNickname even if VMId is provided
        if ($VMId -gt 0) {
            $AgentArgs = "-NoProfile -WindowStyle Hidden -ExecutionPolicy Bypass -File `"$AgentPath`" -BackendUrl `"$BackendUrl`" -VMId $VMId -UnitId `"$UnitId`" -Group `"$Group`" -RdpNickname `"$RdpNickname`""
            Write-Host "      Mode: Update/Heartbeat (VMId: $VMId)" -ForegroundColor Gray
        } else {
            $AgentArgs = "-NoProfile -WindowStyle Hidden -ExecutionPolicy Bypass -File `"$AgentPath`" -BackendUrl `"$BackendUrl`" -Group `"$Group`" -UnitId `"$UnitId`" -RdpNickname `"$RdpNickname`""
            Write-Host "      Mode: Registration (Group: $Group)" -ForegroundColor Gray
        }
        
        $Action = New-ScheduledTaskAction -Execute "powershell.exe" -Argument $AgentArgs
        $Trigger = New-ScheduledTaskTrigger -AtLogon
        $Settings = New-ScheduledTaskSettingsSet -AllowStartIfOnBatteries -DontStopIfGoingOnBatteries -StartWhenAvailable -Hidden
        
        Register-ScheduledTask -Action $Action -Trigger $Trigger -Settings $Settings -TaskName $TaskName -Force -ErrorAction Stop | Out-Null
        Write-Host "      Scheduled task created: $TaskName" -ForegroundColor Green
    } catch {
        Write-Host "      Failed to create scheduled task: $_" -ForegroundColor Red
        $ErrorOccurred = $true
    }
}

# 4. Start Immediately
if (-not $ErrorOccurred) {
    Write-Host "`n[4/4] Starting agent..." -ForegroundColor Yellow
    try {
        Start-ScheduledTask -TaskName $TaskName -ErrorAction Stop
        Write-Host "      Agent started successfully" -ForegroundColor Green
    } catch {
        Write-Host "      Failed to start agent: $_" -ForegroundColor Red
        $ErrorOccurred = $true
    }
}

# Final Status
Write-Host "`n========================================" -ForegroundColor Cyan
if ($ErrorOccurred) {
    Write-Host "  INSTALLATION FAILED" -ForegroundColor Red
    Write-Host "========================================`n" -ForegroundColor Cyan
    exit 1
} else {
    Write-Host "  INSTALLATION SUCCESSFUL" -ForegroundColor Green
    Write-Host "========================================`n" -ForegroundColor Cyan
    Write-Host "  VMId: $VMId" -ForegroundColor Gray
    Write-Host "  Group: $Group" -ForegroundColor Gray
    Write-Host "  Unit ID: $UnitId" -ForegroundColor Gray
    Write-Host "  RDP Name: $RdpNickname" -ForegroundColor Gray
    Write-Host "  Backend: $BackendUrl" -ForegroundColor Gray
    Write-Host "`nThe agent will start automatically on login." -ForegroundColor Green
    exit 0
}