param (
    [string]$BackendUrl = "https://bdma.co.in/earnapp",
    [int]$VMId = 0,
    [string]$Group = "Default",
    [string]$UnitId = "DefaultUnit",
    [string]$RdpNickname = "UnknownRDP"
)

[Net.ServicePointManager]::SecurityProtocol = [Net.SecurityProtocolType]::Tls12
Add-Type -AssemblyName UIAutomationClient, UIAutomationTypes

$LogPath = "$env:APPDATA\SinnerWatchdog\agent.log"
$ConfigPath = "$env:APPDATA\SinnerWatchdog\config.json"

$HeartbeatInterval = 180
$StatsSyncInterval = 21600
$CommandPollInterval = 10

function Write-LocalLog {
    param($msg)
    $ts = Get-Date -Format "yyyy-MM-dd HH:mm:ss"
    if (!(Test-Path (Split-Path $LogPath))) {
        New-Item -ItemType Directory -Path (Split-Path $LogPath) -Force | Out-Null
    }
    "[$ts] $msg" | Out-File $LogPath -Append -Encoding utf8
}

function Get-PublicIP {
    try { (Invoke-RestMethod "https://api.ipify.org?format=json" -TimeoutSec 5).ip }
    catch {
        try { (Invoke-RestMethod "http://ip-api.com/json" -TimeoutSec 5).query }
        catch { "0.0.0.0" }
    }
}

function Save-Config {
    param($Id)
    @{ vm_id = $Id } | ConvertTo-Json | Out-File $ConfigPath -Force
}

function Load-Config {
    if (Test-Path $ConfigPath) {
        (Get-Content $ConfigPath | ConvertFrom-Json).vm_id
    } else { $null }
}

function Get-EarnAppWindow {
    try {
        $ae = [System.Windows.Automation.AutomationElement]::RootElement
        $cond = New-Object System.Windows.Automation.OrCondition(
            (New-Object System.Windows.Automation.PropertyCondition(
                [System.Windows.Automation.AutomationElement]::ProcessIdProperty,
                (Get-Process EarnApp -ErrorAction SilentlyContinue | Select-Object -First 1).Id
            )),
            (New-Object System.Windows.Automation.PropertyCondition(
                [System.Windows.Automation.AutomationElement]::ClassNameProperty,
                "Chrome_WidgetWin_1"
            ))
        )
        return $ae.FindFirst(
            [System.Windows.Automation.TreeScope]::Descendants,
            $cond
        )
    } catch {
        return $null
    }
}
function Get-EarnAppStats {
    Write-LocalLog "Fetching EarnApp stats..."

    $stats = @{ balance = 0; traffic = "0B"; status = "Offline" }
    $win = $null

    try {
        $ae = [System.Windows.Automation.AutomationElement]::RootElement
        $cond = New-Object System.Windows.Automation.PropertyCondition([System.Windows.Automation.AutomationElement]::NameProperty, "EarnApp")
        $win = $ae.FindFirst([System.Windows.Automation.TreeScope]::Children, $cond)
        if ($win) { Write-LocalLog "Attached to existing EarnApp window." }
    } catch {
        $win = $null
    }

    $attempt = 0
    while ($attempt -lt 3 -and -not $win) {
        Stop-Process -Name EarnApp -Force -ErrorAction SilentlyContinue
        Start-Sleep 1

        try { Start-Process "C:\Program Files (x86)\EarnApp\EarnApp.exe" } catch {}

        $retry = 0
        while ($retry -lt 30 -and -not $win) {
            $ae = [System.Windows.Automation.AutomationElement]::RootElement
            $cond = New-Object [System.Windows.Automation.PropertyCondition]([System.Windows.Automation.AutomationElement]::NameProperty, "EarnApp")
            $win = $ae.FindFirst([System.Windows.Automation.TreeScope]::Children, $cond)
            if ($win) { break }
            Start-Sleep 2
            $retry++
        }
        $attempt++
    }

    if (-not $win) {
        Write-LocalLog "EarnApp window still not found"
        return $stats
    }

    Start-Sleep 5 

    try {
        $els = $win.FindAll([System.Windows.Automation.TreeScope]::Descendants, [System.Windows.Automation.Condition]::TrueCondition)
        $balances = @()
        $statusTextFound = $false

        foreach ($e in $els) {
            $t = $e.Current.Name
            
            if ($t -match "\$") {
                $num = ($t -replace '[^0-9\.]', '')
                if ($num) {
                    $balances += [decimal]$num
                }
            }

            if ($t -like "*Earning income*" -or $t -like "*Blocked*") {
                $stats.status = $t
                $statusTextFound = $true
                break 
            }
        }
        
        if ($balances.Count -gt 0) {
            $stats.balance = $balances[-1]
        }

        if (-not $statusTextFound) {
            $stats.status = "Unknown Status (UI Element not found)"
        }

    } catch {
        $stats.status = "UI Error: $($_.Exception.Message)"
    }

    Write-LocalLog "Stats Fetched: Balance = $($stats.balance), Status = $($stats.status)"
    return $stats
}

function Get-EarnAppToken {
    try {
        $path = "$env:LOCALAPPDATA\BrightData"
        if (Test-Path $path) {
            $file = Get-ChildItem $path -File | Select-Object -First 1
            if ($file) {
                $c = (Get-Content $file.FullName -Raw).Trim()
                if ($c.Contains(":")) { ($c -split ":")[0].Trim() } else { $c }
            } else { "Not Found" }
        } else { "Not Found" }
    } catch { "Error" }
}

function Invoke-RestMethodWithRetry {
    param(
        [string]$Uri,
        [string]$Method = "GET",
        [string]$ContentType = "application/json",
        [string]$Body = "",
        [int]$TimeoutSec = 15,
        [int]$MaxRetries = 3,
        [int]$InitialDelaySec = 5
    )
    for ($i = 0; $i -lt $MaxRetries; $i++) {
        try {
            return Invoke-RestMethod -Uri $Uri -Method $Method -ContentType $ContentType -Body $Body -TimeoutSec $TimeoutSec
        } catch {
            $errorMessage = $_.Exception.Message
            Write-LocalLog "Invoke-RestMethod failed (Attempt $($i+1)/$MaxRetries): $errorMessage"
            if ($i -lt ($MaxRetries - 1)) {
                $delay = $InitialDelaySec * [Math]::Pow(2, $i)
                Write-LocalLog "Retrying in $delay seconds..."
                Start-Sleep $delay
            } else {
                Write-LocalLog "Max retries reached. Failing request."
                throw $_.Exception
            }
        }
    }
}

function Send-Data {
    param($id, $stats)
    try {
        Invoke-RestMethodWithRetry "$BackendUrl/save_data.php" -Method POST -ContentType "application/json" -TimeoutSec 15 `
        -Body (@{
            vm_id=$id
            unit_id=$UnitId
            group_name=$Group
            rdp_nickname=$RdpNickname
            balance=("{0:F4}" -f [decimal]$stats.balance)
            traffic=$stats.traffic
            status=$stats.status
            ip_address=(Get-PublicIP)
            earnapp_token=(Get-EarnAppToken)
        } | ConvertTo-Json)
    } catch {
        Write-LocalLog "Sync Error: $($_.Exception.Message)"
    }
}

$CurrentVmId = if ($VMId -ne 0) { $VMId } else { Load-Config }

if (-not $CurrentVmId) {
    while (-not $CurrentVmId) {
        try {
            $r = Invoke-RestMethodWithRetry "$BackendUrl/api/vm/register" -Method POST -ContentType "application/json" -TimeoutSec 15 `
            -Body (@{
                group_name=$Group
                rdp_nickname=$RdpNickname
                unit_id=$UnitId
                assigned_ip=(Get-PublicIP)
            } | ConvertTo-Json)
            $CurrentVmId = $r.vm_id
            Save-Config $CurrentVmId
        } catch {
            Write-LocalLog "VM registration failed: $($_.Exception.Message)"
            Start-Sleep 10
        }
    }
}

$lastHeartbeat = Get-Date 0
$lastStatsSync = Get-Date 0

while ($true) {
    Start-Sleep 5
    $now = Get-Date

    if (($now - $lastHeartbeat).TotalSeconds -ge $HeartbeatInterval) {
        try {
            Invoke-RestMethodWithRetry "$BackendUrl/save_data.php" -Method POST -ContentType "application/json" -TimeoutSec 10 `
            -Body (@{ action="heartbeat"; vm_id=$CurrentVmId; ip_address=(Get-PublicIP) } | ConvertTo-Json)
            Write-LocalLog "Heartbeat sent"
        } catch {
            Write-LocalLog "Heartbeat failed: $($_.Exception.Message)"
        }
        $lastHeartbeat = $now
    }

    if (($now - $lastStatsSync).TotalSeconds -ge $StatsSyncInterval) {
        $stats = Get-EarnAppStats
        Send-Data $CurrentVmId $stats
        $lastStatsSync = $now
    }
}