<?php
session_start();
include '../db.php';  // Database connection

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$userId = $_SESSION['user_id'];
$stmt = $conn->prepare("SELECT * FROM users WHERE id = ?");
$stmt->bind_param("i", $userId);
$stmt->execute();
$userResult = $stmt->get_result();
$user = $userResult->fetch_assoc();
$stmt->close();


//======================================================================
// GOOGLE SHEET SYNCING CODE - START
//======================================================================

/**
 * Sends a BATCH of records to the Google Apps Script Web App.
 *
 * @param array $records An array of associative arrays, where each inner array is a row.
 * @return bool True on success, false on failure.
 */
function sendBatchToGoogleSheet($records) {
    if (empty($records)) {
        return true; // Nothing to send, so it's a success.
    }

    $googleUrl = "https://script.google.com/macros/s/AKfycbwxIQXrgsL2BXjWyjWCWBRcmjAZINef0W8wlt-YBHaU1_Q2PPwJiTRhzSwudOwxYrRi/exec";
    $payload = json_encode(['records' => $records]);

    $ch = curl_init($googleUrl);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, false); // Crucial: Do NOT follow the redirect.
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);

    curl_exec($ch); // We execute the request but don't need the HTML response body.
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);

    if ($error) {
        error_log("cURL Error sending to Google Sheet: " . $error);
        return false;
    }

    // THE FINAL FIX: A 302 status code is a SUCCESS for a Google Apps Script doPost.
    // This means the script ran and is now redirecting. We treat this as success.
    if ($httpCode !== 302) {
        error_log("Google Sheet API Error: Expected HTTP status 302, but got " . $httpCode);
        return false;
    }

    // If the code is 302, it was successful.
    return true;
}

// --- Main Logic for Syncing New Records ---
$trackerFile = 'last_sent_id.txt';

// Use file locking to prevent race conditions if the script runs multiple times quickly.
$fp = fopen($trackerFile, 'c+');
if ($fp && flock($fp, LOCK_EX)) {

    $lastSentId = (int)fread($fp, 1024);
    $newestIdInBatch = $lastSentId;
    $recordsToSend = [];

    // Fetch only new records from the database
    $newRecordsSql = "SELECT * FROM applications WHERE id > ? ORDER BY id ASC LIMIT 100"; // Limit to 100 per run for safety
    $stmt = $conn->prepare($newRecordsSql);
    $stmt->bind_param("i", $lastSentId);
    $stmt->execute();
    $newRecordsResult = $stmt->get_result();

    if ($newRecordsResult && $newRecordsResult->num_rows > 0) {
        while ($row = $newRecordsResult->fetch_assoc()) {
            // Map database columns to the keys expected by the Google Sheet
            $recordsToSend[] = [
                'id' => $row['id'],
                'full_name' => $row['full_name'],
                'dob' => $row['dob'],
                'gender' => $row['gender'],
                'email' => $row['email'],
                'phone' => $row['phone'],
                'nationality' => $row['nationality'],
                'perm_street' => $row['perm_street'],
                'perm_state' => $row['perm_state'],
                'perm_city' => $row['perm_city'],
                'license_number' => $row['license_number'],
                'plan' => $row['plan'],
                'type' => $row['type'],
                'license_country' => $row['license_country'],
                'drive_country' => $row['drive_country'],
                'addons' => $row['addons'],
                'promo' => $row['promo'],
                'ship_fname' => $row['ship_fname'],
                'ship_lname' => $row['ship_lname'],
                'street' => $row['street'],
                'state' => $row['state'],
                'zip' => $row['zip'],
                'city' => $row['city'],
                'country' => $row['country'],
                'created_at' => $row['created_at'],
                'product' => $row['product'],
                'validity' => $row['validity'],
                'order_id' => $row['order_id'],
                'total_price' => $row['total_price']
            ];
            $newestIdInBatch = $row['id']; // Keep track of the highest ID in this batch
        }

        // Now send the entire batch to Google Sheets in one go
        if (sendBatchToGoogleSheet($recordsToSend)) {
            // If sending was successful, update the tracker file with the newest ID
            ftruncate($fp, 0);      // Clear the file
            rewind($fp);            // Go to the beginning
            fwrite($fp, $newestIdInBatch); // Write the new last ID
        }
    }
    $stmt->close();
    
    flock($fp, LOCK_UN); // Release the lock
    fclose($fp);
}

//======================================================================
// GOOGLE SHEET SYNCING CODE - END
//======================================================================


// Helper to show images safely
function showImage($file) {
    // Using basename for security to prevent path traversal
    $safeFile = basename($file);
    if (!empty($safeFile) && file_exists("../uploads/$safeFile")) {
        return "<img src='../uploads/$safeFile' width='50' alt='Application Document'>";
    }
    return "<span class='text-danger'>Missing</span>";
}

// Fetch all applications to display on the page
$sql = "SELECT * FROM applications ORDER BY created_at DESC";
$result = $conn->query($sql);

?> 

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    
    <title>Itca Dashboard</title>
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.6/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.6/dist/js/bootstrap.bundle.min.js"></script>
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .circle-name { 
            width: 40px; height: 40px; 
            border-radius: 50%; background: #007bff; 
            color: #fff; text-align: center; line-height: 40px; 
            font-weight: bold; margin-right: 10px;
        }
        .table-style img { max-height: 50px; }
    </style>
</head>
<body>
<div class="dashboard_wrapper">
    <div class="left_panel">
        <div class="logo_box">
            <img src="../assets/images/logo.png" alt="logo" class="img-fluid full_logo">
            <img src="../assets/images/logo-icon.png" alt="logo" class="img-fluid logo_icon">
        </div>
        <?php include 'menu.php'; ?>
    </div>

        <div class="right_body_panel">
        <div class="right_header_panel">
            <div class="left_toggle_button">
                <button class="btn toggle_btn" type="button"></button>
            </div>
            <div class="right_user_box">
                <div class="user_box_warp">
                    <span class="circle-name"><?php echo strtoupper(substr($user['full_name'], 0, 2)); ?></span>
                    <div class="user_name"><h4><?php echo htmlspecialchars($user['full_name']); ?></h4></div>
                </div>
            </div>
         
 </div>

        <div class="body_wrapper_section p-3">
            <h3>Permit Request(s)</h3>
            <div class="table-wrapper inner-wrapper">
                <table id="myTable" class="display table table-bordered table-striped table-style">
                    <button id="downloadAll" class="btn btn-success btn-sm">Download All CSV</button>
<button id="downloadSelected" class="btn btn-primary btn-sm">Download Selected CSV</button>
                    <thead>
<tr>
    <th><input type="checkbox" id="selectAll"></th>
    <th>ID</th>
    <th>Full Name</th>
    <th>DOB</th>
    <th>Gender</th>
    <th>Email</th>
    <th>Phone</th>
    <th>Nationality</th>
    <th>Street</th>
    <th>State</th>
    <th>City</th>
    <th>License Number</th>
    <th>Plan</th>
    <th>Type</th>
    <th>License Country</th>
    <th>Drive Country</th>
    <th>License Class</th>
    <th>Addons</th>
    <th>Promo</th>
    <th>Passport</th>
    <th>License Front</th>
    <th>License Back</th>
    <th>Ship First</th>
    <th>Ship Last</th>
    <th>Street</th>
    <th>State</th>
    <th>ZIP</th>
    <th>City</th>
    <th>Country</th>
    <th>Created At</th>
    <th>Product</th>
    <th>Validity</th>
    <th>GRM ID</th>
    <th>Prices</th>
    <th colspan="3" align="center">Action</th>
</tr>


                    </thead>
                    <tbody>
                        
                        <?php
                        if ($result->num_rows > 0) {
                            while ($row = $result->fetch_assoc()) {
                                $licenseClassHtml = str_replace('src="assets/', 'src="../assets/', $row['license_class']);
                                $licenseClassHtml = str_replace('Â', '', $licenseClassHtml);
                                
                                 
                                $imgSplit = explode('<img', $licenseClassHtml);
                                if (count($imgSplit) > 2) {
                                   
                                    $licenseClassHtml = $imgSplit[0] . '<img' . $imgSplit[1] . '<br><img' . $imgSplit[2];
                                
                                     
                                    for ($i = 3; $i < count($imgSplit); $i++) {
                                        $licenseClassHtml .= '<img' . $imgSplit[$i];
                                    }
                                }
                                echo "<tr>";
                                echo "<td><input type='checkbox' class='selectRow' value='{$row['id']}'></td>";
                                echo "<td>{$row['id']}</td>";
                                echo "<td>{$row['full_name']}</td>";
                                echo "<td>{$row['dob']}</td>";
                                echo "<td>{$row['gender']}</td>";
                                echo "<td>{$row['email']}</td>";
                                echo "<td>{$row['phone']}</td>";
                                echo "<td>{$row['nationality']}</td>";
                                echo "<td>{$row['perm_street']}</td>";
                                echo "<td>{$row['perm_state']}</td>";
                                echo "<td>{$row['perm_city']}</td>";
                                echo "<td>{$row['license_number']}</td>";
                                echo "<td>{$row['plan']}</td>";
                                echo "<td>{$row['type']}</td>";
                                echo "<td>{$row['license_country']}</td>";
                                echo "<td>{$row['drive_country']}</td>";
                                echo "<td><div class='value'>$licenseClassHtml</div></td>";
                                $addons = explode(',', $row['addons']);  
                                echo "<td>";
                                foreach ($addons as $addon) {
                                    list($product, $image) = explode('|', $addon);  
                                    echo htmlspecialchars($product) . "<br>"; 
                                }
                                echo "</td>";
                                echo "<td>" . htmlspecialchars($row['promo']) . "</td>";
                                echo "<td>" . showImage($row['passport_photo']) . "</td>";
                                echo "<td>" . showImage($row['license_front']) . "</td>";
                                echo "<td>" . showImage($row['license_back']) . "</td>";
                                echo "<td>{$row['ship_fname']}</td>";
                                echo "<td>{$row['ship_lname']}</td>";
                                echo "<td>" . (!empty($row['street']) ? htmlspecialchars($row['street']) : "<span class='text-muted'>Not provided</span>") . "</td>";
                                echo "<td>{$row['state']}</td>";
                                echo "<td>{$row['zip']}</td>";
                                echo "<td>{$row['city']}</td>";
                                echo "<td>{$row['country']}</td>";
                                echo "<td>{$row['created_at']}</td>";
                                echo "<td>{$row['product']}</td>";
                                echo "<td>{$row['validity']}</td>";
                                echo "<td>{$row['order_id']}</td>";
                                echo "<td>" . htmlspecialchars($row['total_price']) . "</td>";
                                echo "<td>
                                    <a href='detail.php?id=" . $row['id'] . "' class='btn btn-warning btn-sm'>View</a>
                                </td>";
                                echo "<td>
                                <a href='view_details.php?id=" . $row['id'] . "' class='btn btn-warning'>Edit</a> </td>";
                                 echo "<td>
                                <button class='btn btn-danger btn-sm removeBtn' data-id='{$row['id']}'>Remove</button>
                              </td>";
                                echo "</tr>";
                            }
                        } else {
                            echo "<tr><td colspan='32'>No records found</td></tr>";
                        }
                        ?>
                    </tbody>
                </table>
            </div>
        </div>

        <!-- Modal -->
        <div class="modal fade" id="applicationModal" tabindex="-1">
            <div class="modal-dialog modal-xl modal-dialog-scrollable">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title">Application ID: <span id="modalAppId"></span></h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                    </div>
                    <div class="modal-body" id="modalContent"></div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    </div>
                </div>
            </div>
        </div>

    </div>
</div>

<script>
document.addEventListener("DOMContentLoaded", function () {
    const modal = new bootstrap.Modal(document.getElementById('applicationModal'));

    /* ----------------------------
       VIEW BUTTON (Existing)
    ---------------------------- */
    document.querySelectorAll('.viewBtn').forEach(btn => {
        btn.addEventListener('click', function () {
            const row = this.closest('tr');
            const cells = row.querySelectorAll('td');
            let html = `
                <h4>Applicant Details</h4>
                <p><strong>Full Name:</strong> ${cells[1].innerHTML}</p>
                <p><strong>DOB:</strong> ${cells[2].innerHTML}</p>
                <p><strong>Gender:</strong> ${cells[3].innerHTML}</p>
                <p><strong>Email:</strong> ${cells[4].innerHTML}</p>
                <p><strong>Phone:</strong> ${cells[5].innerHTML}</p>
                <p><strong>Nationality:</strong> ${cells[6].innerHTML}</p>
                <p><strong>License Number:</strong> ${cells[7].innerHTML}</p>

                <h4>License & Plan</h4>
                <p><strong>Plan:</strong> ${cells[8].innerHTML}</p>
                <p><strong>Type:</strong> ${cells[9].innerHTML}</p>
                <p><strong>License Country:</strong> ${cells[10].innerHTML}</p>
                <p><strong>Drive Country:</strong> ${cells[11].innerHTML}</p>
                <p><strong>License Class:</strong> ${cells[12].innerHTML}</p>
                <p><strong>Addons:</strong> ${cells[13].innerHTML}</p>
                <p><strong>Promo:</strong> ${cells[14].innerHTML}</p>

                <h4>Documents</h4>
                <p><strong>Passport:</strong> ${cells[15].innerHTML}</p>
                <p><strong>License Front:</strong> ${cells[16].innerHTML}</p>
                <p><strong>License Back:</strong> ${cells[17].innerHTML}</p>

                <h4>Shipping Address</h4>
                <p><strong>Name:</strong> ${cells[18].innerHTML} ${cells[19].innerHTML}</p>
                <p><strong>Street:</strong> ${cells[20].innerHTML}</p>
                <p><strong>State:</strong> ${cells[21].innerHTML}</p>
                <p><strong>ZIP:</strong> ${cells[22].innerHTML}</p>
                <p><strong>City:</strong> ${cells[23].innerHTML}</p>
                <p><strong>Country:</strong> ${cells[24].innerHTML}</p>
                
                <h4>Other</h4>
                <p><strong>Created At:</strong> ${cells[25].innerHTML}</p>
                <p><strong>Product:</strong> ${cells[26].innerHTML}</p>
                <p><strong>Validity:</strong> ${cells[27].innerHTML}</p>
                <p><strong>Tracking:</strong> ${cells[2].innerHTML}</p>
                <p><strong>Prices:</strong> ${cells[31].innerHTML}</p>
            `;

            document.getElementById('modalAppId').textContent = cells[0].innerHTML;
            document.getElementById('modalContent').innerHTML = html;
            modal.show();
        });
    });

    /* ----------------------------
       REMOVE BUTTON (Existing)
    ---------------------------- */
    document.querySelectorAll('.removeBtn').forEach(btn => {
        btn.addEventListener('click', function () {
            const id = this.dataset.id;
            if (confirm('Are you sure you want to remove this application?')) {
                fetch('delete_application.php?id=' + id, { method: 'GET' })
                    .then(response => response.json())
                    .then(data => {
                        if (data.success) {
                            alert('Application removed successfully.');
                            location.reload();
                        } else {
                            alert('Error: ' + data.message);
                        }
                    })
                    .catch(err => alert('Error: ' + err));
            }
        });
    });


    /* ----------------------------
       CSV DOWNLOAD FEATURE (NEW)
    ---------------------------- */

    // Select All checkbox
    const selectAll = document.getElementById("selectAll");
    if (selectAll) {
        selectAll.addEventListener("click", function () {
            document.querySelectorAll(".selectRow").forEach(cb => cb.checked = this.checked);
        });
    }

    // Download all records
    const downloadAll = document.getElementById("downloadAll");
    if (downloadAll) {
        downloadAll.addEventListener("click", function () {
            window.location.href = "download_csv.php?mode=all";
        });
    }

    // Download selected records
    const downloadSelected = document.getElementById("downloadSelected");
    if (downloadSelected) {
        downloadSelected.addEventListener("click", function () {
            const selected = Array.from(document.querySelectorAll(".selectRow:checked")).map(cb => cb.value);
            if (selected.length === 0) {
                alert("Please select at least one record.");
                return;
            }

            const form = document.createElement("form");
            form.method = "POST";
            form.action = "download_csv.php";
            form.style.display = "none";

            const input = document.createElement("input");
            input.name = "ids";
            input.value = selected.join(",");
            form.appendChild(input);

            document.body.appendChild(form);
            form.submit();
        });
    }

});
</script>


</body>
</html>
