const safeSetText = (id, value) => {
    const el = document.getElementById(id);
    if (el) el.innerText = value;
};
const safeSetValue = (id, value) => {
    const el = document.getElementById(id);
    if (el) el.value = value;
};

let promoInfo = window.promoInfo || { code: '', type: '', value: 0, promo_id: null, error: '' };
if (typeof window.promoInfo === 'undefined') {
    window.promoInfo = promoInfo;
}

function handleCountryChange() {
    const conventionDisplay = document.getElementById('idp-convention-display');
    if (!conventionDisplay) return;

    const selectedOption = this.options[this.selectedIndex];
    const conventionText = selectedOption.dataset.convention;

    if (conventionText && conventionText.trim() !== '') {
        conventionDisplay.innerHTML = `<br>${conventionText.replace(/\n/g, '<br>')}`;
        conventionDisplay.style.display = 'block';
    } else {
        conventionDisplay.style.display = 'none';
        conventionDisplay.innerHTML = '';
    }
}

function populateCountries() {
    const licenseIssueSelect = document.getElementById("license_issue_label");
    const driveCountrySelect = document.getElementById("drive_country");

    if (!licenseIssueSelect || !driveCountrySelect || !window.countriesData) {
        return;
    }

    licenseIssueSelect.innerHTML = '<option value="">Select Country</option>';
    driveCountrySelect.innerHTML = '<option value="">Select Country</option>';

    window.countriesData.forEach(country => {
        const option = document.createElement('option');
        option.value = country.country_code;
        option.textContent = `${country.country_name} (${country.country_code})`;
        option.dataset.convention = country.idp_convention || '';

        licenseIssueSelect.appendChild(option.cloneNode(true));
        driveCountrySelect.appendChild(option.cloneNode(true));
    });

    licenseIssueSelect.addEventListener('change', handleCountryChange);
}


function updateSectionVisibility() {
    const copyType = document.querySelector('input[name="plan"]:checked')?.value;
    const productSelected = document.querySelector('input[name="productbooklet"]:checked')?.value;
    const validitySection = document.getElementById('validitySection');
    const addonsSection = document.getElementById('addonsSection');

    if (!validitySection || !addonsSection) {
        return;
    }

    const shouldShow = (copyType === 'Soft Copy' || copyType === 'Hard Copy') && (productSelected === 'Booklet' || productSelected === 'Booklet + Card');

    if (shouldShow) {
        validitySection.style.display = 'block';
        addonsSection.style.display = 'block';
    } else {
        validitySection.style.display = 'none';
        addonsSection.style.display = 'none';
        document.getElementById('validity').value = '';
        document.getElementById('addonsContainer').innerHTML = '';
    }
}

function updateValidityOptions() {
    const validitySelect = document.getElementById('validity');
    if (!validitySelect) return;

    const copyType = document.querySelector('input[name="plan"]:checked')?.value;
    const productSelected = document.querySelector('input[name="productbooklet"]:checked')?.value;
    const structuredData = window.prices?.structured_data || {};

    if (!copyType || !productSelected || !structuredData[productSelected]) {
        validitySelect.innerHTML = '<option value="">Select Validity</option>';
        return;
    }

    const productData = structuredData[productSelected];
    const copyKey = (copyType === 'Soft Copy') ? 'soft' : 'hard';

    const currentValidity = validitySelect.value;
    validitySelect.innerHTML = '<option value="">Select Validity</option>';

    Object.keys(productData).forEach(validityId => {
        const validity = productData[validityId];
        const addon = validity.addons?.[copyKey];
        if (addon && addon.price > 0) {
            const option = document.createElement('option');
            option.value = validityId;
            option.text = `Validity ${validity.validity_name} Years`;
            option.dataset.price = addon.price;
            validitySelect.appendChild(option);
        }
    });
    if (currentValidity) validitySelect.value = currentValidity;
}

function updateAddonsOptions() {
    const addonsContainer = document.getElementById('addonsContainer');
    if (!addonsContainer) return;

    const copyType = document.querySelector('input[name="plan"]:checked')?.value;
    const productSelected = document.querySelector('input[name="productbooklet"]:checked')?.value;
    const additionalAddons = window.prices?.additional_addons || [];

    addonsContainer.innerHTML = "";

    if (!copyType || !productSelected) {
        return;
    }

    additionalAddons.forEach(addon => {
        let price = 0;
        let name = addon.name;

        if (copyType === 'Soft Copy') {
            name = addon.addon_soft_product_name || name;
            if (productSelected === 'Booklet') {
                price = parseFloat(addon.addon_soft_booklet_only_price);
            } else if (productSelected === 'Booklet + Card') {
                price = parseFloat(addon.addon_soft_booklet_card_price);
            }
        } else if (copyType === 'Hard Copy') {
            if (productSelected === 'Booklet') {
                price = parseFloat(addon.addon_hard_booklet_only_price);
            } else if (productSelected === 'Booklet + Card') {
                price = parseFloat(addon.addon_hard_booklet_card_price);
            }
        }

        if (price > 0 && name) {
            const wrapper = document.createElement('div');
            const checkbox = document.createElement('input');
            checkbox.type = 'checkbox';
            checkbox.name = 'addons[]';
            checkbox.id = `addon_${addon.id}`;
            checkbox.value = addon.id;
            checkbox.dataset.price = price;
            checkbox.addEventListener('change', updateTotal);

            const label = document.createElement('label');
            label.setAttribute('for', checkbox.id);
            label.textContent = ` ${name}`;

            wrapper.appendChild(checkbox);
            wrapper.appendChild(label);
            addonsContainer.appendChild(wrapper);
        }
    });
}

function checkUrgentAvailability() {
    const copyType = document.querySelector('input[name="plan"]:checked')?.value;
    const urgentRadio = document.getElementById('urgentRadio');
    const normalRadio = document.querySelector('input[name="type"][value="Normal"]');
    if (!copyType || !urgentRadio || !normalRadio) return;

    const ukHour = window.ukHour;
    const ukDay = window.ukDay;

    const isWeekday = ukDay >= 1 && ukDay <= 5;
    const isWeekend = ukDay === 0 || ukDay === 6;
    const isWorkingHours = ukHour >= 10 && ukHour < 17;

    let enableUrgent = false;
    if (copyType === 'Soft Copy') enableUrgent = isWorkingHours;
    else if (copyType === 'Hard Copy') enableUrgent = isWeekday && isWorkingHours;

    urgentRadio.disabled = !enableUrgent;
    if (!enableUrgent && urgentRadio.checked) {
        urgentRadio.checked = false;
        normalRadio.checked = true;
    }

    let currentPrice = 0;
    const selectedType = document.querySelector('input[name="type"]:checked')?.value;
 
    if (selectedType === 'Urgent' && enableUrgent) {
        if (copyType === 'Soft Copy') {
            // Soft + Urgent (Weekday / Weekend)
            currentPrice = isWeekend
                ? parseFloat(window.prices.weekend_type_price)
                : parseFloat(window.prices.urgent_type_price);
        } else if (copyType === 'Hard Copy') {
            // Hard + Urgent (Weekday only)
            currentPrice = parseFloat(window.prices.normal_type_price);
        }
    } else {
        
        if (copyType === 'Soft Copy') {
            currentPrice = parseFloat(window.prices.soft_copy_price);
        } else if (copyType === 'Hard Copy') {
            currentPrice = parseFloat(window.prices.hard_copy_price);
        }
    }

    
    window.prices.currentTypePrice = currentPrice;
    updateTotal();
}

function updateTotal() {
    let validityPrice = 0, typePrice = 0, addOnsPrice = 0, licenseClassesPrice = 0, discountAmount = 0;

    const validitySelect = document.getElementById('validity');
    const selectedValidityOption = validitySelect?.selectedOptions[0];
    if (selectedValidityOption && selectedValidityOption.value) {
        validityPrice = parseFloat(selectedValidityOption.dataset.price || 0);
    }

    if (document.getElementById('urgentRadio')?.checked) {
        typePrice = parseFloat(window.prices?.currentTypePrice || 0);
    }

    document.querySelectorAll('input[name="addons[]"]:checked').forEach(el => {
        addOnsPrice += parseFloat(el.dataset.price || 0);
    });

    document.querySelectorAll('input[name="license_class[]"]:checked').forEach(el => {
        licenseClassesPrice += parseFloat(el.dataset.price || 0);
    });

    let subtotal = validityPrice + typePrice + addOnsPrice + licenseClassesPrice;

    if (promoInfo && promoInfo.value > 0 && promoInfo.type) {
        discountAmount = (promoInfo.type === 'percent') ? (subtotal * promoInfo.value / 100) : parseFloat(promoInfo.value);
        discountAmount = Math.min(Math.max(discountAmount, 0), subtotal);
    }

    safeSetText("breakdown-validity-price", "£" + validityPrice.toFixed(2));
    safeSetText("breakdown-type-price", "£" + typePrice.toFixed(2));
    safeSetText("breakdown-addons-price", "£" + addOnsPrice.toFixed(2));
    safeSetText("breakdown-license-classes-price", "£" + licenseClassesPrice.toFixed(2));
    safeSetText("breakdown-discount", "-£" + discountAmount.toFixed(2));

    let total = subtotal - discountAmount;
    safeSetText("totalAmount", "£" + total.toFixed(2));
    safeSetValue("totalInput", total.toFixed(2));
}

function masterUpdate() {
    updateSectionVisibility();
    updateValidityOptions();
    updateAddonsOptions();
    checkUrgentAvailability();
    updateTotal();
}

document.addEventListener('DOMContentLoaded', () => {
    populateCountries();
    document.addEventListener('DOMContentLoaded', checkUrgentAvailability);
    setInterval(checkUrgentAvailability, 60 * 1000);

    fetch('get_validity_prices.php')
        .then(res => res.json())
        .then(data => {
            window.prices = window.prices || {};
            window.prices.structured_data = data.validity_data;
            window.prices.additional_addons = data.additional_addons;

            document.querySelectorAll('input[name="plan"], input[name="productbooklet"], input[name="type"], input[name="license_class[]"]').forEach(el => {
                el.addEventListener('change', masterUpdate);
            });
            document.getElementById('validity').addEventListener('change', updateTotal);

            masterUpdate();
        })
        .catch(err => console.error("Failed to load combined data:", err));
});
